(ns codescene.features.api.routes.developer-settings
  (:require [compojure.api.sweet :refer [context routes GET POST PUT DELETE]]
            [spec-tools.spec :as spec]
            [spec-tools.data-spec :as ds]
            [codescene.features.api.privileges :as api-privileges]
            [codescene.features.api.developer-settings :as api-developer-settings]
            [codescene.features.components.auth :as auth]
            [codescene.features.components.developer-settings :as developer-settings]
            [codescene.features.api.teams :as api-teams]
            [codescene.features.api.core :as api-core]
            [codescene.features.api.developers :as api-developers]
            [codescene.features.api.spec.analysis :as api-analysis-spec]
            [codescene.features.util.api :as api-utils]
            [codescene.presentation.display :as display]
            [clojure.spec.alpha :as s]
            [medley.core :as m]))

(defn- verify-access [system developer-settings-id handler]
  (fn [request]
    (let [user (auth/user (api-core/api-auth system) request)
          developer-setting? (->> (developer-settings/developer-settings-list (api-core/api-developer-settings system) user)
                                  (m/find-first #(= (:id %) (display/->maybe-int developer-settings-id))))]
      (if developer-setting?
        (handler request)
        (api-utils/not-found (format "There's no developer-settings with id: %s or you do not have access to it" developer-settings-id))))))

(defn- sub-routes-reads
  [system]
  (context "/developer-settings" [] ;; Read routes
    :query-params []
    :middleware [#(api-core/wrap-authorize-any system % #{api-privileges/restapi-read})]
    (GET "/" request
      :summary "Developer Settings List"
      :tags ["teams and developers"]
      :query-params [{page :- spec/int? 1}]
      :responses {200 {:schema      ::api-analysis-spec/developer_settings_list
                       :description "developer settings list"}}
      (api-developer-settings/list-result system request page))
    (context "/:developer_setting_id" [developer_setting_id]
      :path-params [developer_setting_id :- spec/pos-int?]
      :middleware [#(verify-access system developer_setting_id %)]
      (GET "/teams" []
        :summary "Teams list"
        :tags ["teams and developers"]
        :query-params [{page :- spec/int? 1}]
        :responses {200 {:schema      ::api-analysis-spec/teams_list
                         :description "teams list for the given developer_setting_id"}}
        (api-teams/list-result system developer_setting_id page))
      (GET "/developers" []
        :summary "Developers list"
        :tags ["teams and developers"]
        :query-params [{page :- spec/int? 1}]
        :responses {200 {:schema      ::api-analysis-spec/developers_list
                         :description "developers list for the given developer_setting_id"}}
        (api-developers/list-result system developer_setting_id page)))))

(defn- teams-routes-writes
  [system developer_setting_id]
  (routes
    (POST "/teams/new" []
      :summary "Create new team"
      :tags ["teams and developers"]
      :body [data (ds/spec {:spec (s/keys :req-un [::api-analysis-spec/team_name]) :name :team})]
      :responses {201 {:schema      spec/any?
                       :description "the team json data"}}
      (api-teams/post-new-team system developer_setting_id data))
    (PUT "/teams/:team_id" [team_id]
      :path-params [team_id :- spec/pos-int?]
      :summary "Update a team"
      :tags ["teams and developers"]
      :body [data (ds/spec {:spec (s/keys :req-un [::api-analysis-spec/team_name]) :name :team})]
      :responses {201 {:schema      spec/any?
                       :description "the team json data"}}
      (api-teams/update-team system developer_setting_id team_id data))
    (DELETE "/teams/:team_id" [team_id]
      :path-params [team_id :- spec/pos-int?]
      :summary "Delete a team"
      :tags ["teams and developers"]
      :responses {200 {:schema      spec/any?
                       :description "delete the team with the given team_id"}}
      (api-teams/delete-team system developer_setting_id team_id))))

(defn- developers-routes-writes
  [system developer_setting_id]
  (routes
    (PUT "/developers/:developer_id" [developer_id]
      :path-params [developer_id :- spec/pos-int?]
      :summary "Update a developer"
      :tags ["teams and developers"]
      :body [data (ds/spec {:spec (s/keys :opt-un [::api-analysis-spec/team_id ::api-analysis-spec/former_contributor ::api-analysis-spec/exclude_from_all_analyses]) :name :developer})]
      :responses {201 {:schema      spec/any?
                       :description "the developer json data, you can use any combination of team_id, former_contributor and exclude_from_all_analyse"}}
      (api-developers/update-developer system developer_setting_id developer_id data))
    (DELETE "/developers/:developer_id" [developer_id]
      :path-params [developer_id :- spec/pos-int?]
      :summary "Delete a developer"
      :tags ["teams and developers"]
      :responses {200 {:schema      spec/any?
                       :description "delete the developer with the given developer_id"}}
      (api-developers/delete-developer system developer_setting_id developer_id))
    (DELETE "/developers/:developer_id/team" [developer_id]
      :path-params [developer_id :- spec/pos-int?]
      :summary "Remove a developer from team"
      :tags ["teams and developers"]
      :responses {200 {:schema      spec/any?
                       :description "remove the developer from it's team"}}
      (api-developers/delete-developer-team system developer_setting_id developer_id))))

(defn- sub-routes-writes
  [system]
  (context "/developer-settings" [] ;; Write routes
    :query-params []
    :middleware [#(api-core/wrap-authorize-any system % #{api-privileges/restapi-write})]
    (context "/:developer_setting_id" [developer_setting_id]
      :path-params [developer_setting_id :- spec/pos-int?]
      :middleware [#(verify-access system developer_setting_id %)]
      (DELETE "/" []
        :summary "Delete a developer settings"
        :tags ["teams and developers"]
        :responses {200 {:schema      spec/any?
                         :description "delete the developer settings with the given developer_setting_id"}}
        (api-developer-settings/delete-result system developer_setting_id))
      (teams-routes-writes system developer_setting_id)
      (developers-routes-writes system developer_setting_id))))

(defn sub-routes
  [system]
  (routes
    (sub-routes-reads system)
    (sub-routes-writes system)))
