(ns codescene.features.api.routes.code-coverage
  (:require [clojure.spec.alpha :as s]
            [codescene.features.api.code-coverage :as code-coverage]
            [codescene.features.api.privileges :as api-privileges]
            [codescene.features.api.core :as api-core]
            [codescene.features.api.spec.code-coverage :as api-code-coverage-spec]
            [codescene.features.components.auth :as auth]
            [compojure.api.sweet :refer [routes GET POST PUT DELETE context]]
            [spec-tools.data-spec :as ds]
            [spec-tools.spec :as spec]))

(defn- user
  [system request]
  (auth/user (api-core/api-auth system) request))

(defn- code-coverage-read-routes
  [system]
  (context "/" []
    :middleware [#(api-core/wrap-authorize-any system % #{api-privileges/restapi-read api-privileges/cli-access})]
    (GET "/" req
      :summary "Get metadata for uploaded code coverage data"
      :tags ["code-coverage"]
      :query-params [{repo-url :- ::api-code-coverage-spec/repo-url ""}]
      :responses {200 {:schema      spec/any?
                       :description "metadata for uploaded code coverage data"}}
      (code-coverage/get-data system (user system req) {:repo-url repo-url}))))

(defn- code-coverage-write-routes
  [system]
  (context "/" []
    ;;we need to give access to restapi users and to cli users
    :middleware [#(api-core/wrap-authorize-any system % #{api-privileges/restapi-write api-privileges/cli-access})]
    (POST "/upload" req
      :summary "Request uploading code coverage data"
      :tags ["code-coverage"]
      :body [data (ds/spec {:name :metadata
                            :spec (s/keys :req-un [::api-code-coverage-spec/repo-url
                                                   ::api-code-coverage-spec/commit-sha
                                                   ::api-code-coverage-spec/format
                                                   ::api-code-coverage-spec/metric]
                                          :opt-un [::api-code-coverage-spec/repo-path
                                                   ::api-code-coverage-spec/subpath])})]
      :responses {200 {:schema      ::api-code-coverage-spec/upload-id
                       :description "an id to use when uploading"}}
      (code-coverage/request-upload system (user system req) data))
    (PUT "/upload/:upload-id" req
      :summary "Upload code coverage data"
      :tags ["code-coverage"]
      :responses {201 {:schema      spec/any?
                       :description "upload code coverage data"}}
      :path-params [upload-id :- spec/pos-int?]
      (code-coverage/upload-data system (user system req) upload-id (:body req)))
    (DELETE "/" req
      :summary "Delete code coverage data"
      :tags ["code-coverage"]
      :body [query-params (ds/spec {:name :query-params
                                    :spec (s/keys :req-un [::api-code-coverage-spec/repo-url])})]
      :responses {201 {:schema      spec/any?
                       :description "delete code coverage data"}}
      (code-coverage/delete-data system (user system req) query-params))))

(defn sub-routes
  [system]
  (context "/code-coverage" []
    :middleware [#(api-core/wrap-can-use-feature system % :code-coverage?)]
    (code-coverage-read-routes system)
    (code-coverage-write-routes system)))
