// @flow
import $ from 'jquery';
import 'tablesorter';

import * as tableSorter from 'codescene-ui/src/table-sorter';

/**
 * Deletes the given developer. Returns a promise of whether or not the developer was
 * deleted, as the user might abort.
 */
function deleteDeveloper(settingsId, devId, devName) : Promise<boolean> {
  if (!confirm('Are you sure you want to permanently delete the Developer ' + devName + '?')) {
    return Promise.resolve(false);
  }

  return new Promise((resolve, reject) => {
    $.ajax({
      type: 'POST',
      url: `${window.baseUrl}developer-configurations/${settingsId}/developers/delete`,
      data: { devid: devId },
      success: function () {
        resolve(true);
      },
      error: function (xhr) {
        reject(xhr.responseText);
      }
    });
  });
}

export function init(settingsId: number, hasTeams : boolean) {

  const table = $('#all-developers-table');

  table.find('.delete-developer-button')
    .on('click', function () {
      const devId = $(this).data('dev-id');
      const devName = $(this).data('dev-name');
      deleteDeveloper(settingsId, devId, devName)
        .then(deleted => {
          if (deleted) {
            $('tr[data-dev-id=' + devId + ']').remove();
          }
        })
        .catch((err) => {
          alert('Failed to delete the Developer ' + devName + '. Reason: ' + err);
        });
      return false;
    });

  $.tablesorter.addParser({
    id: 'select',
    is: function () {
      return false;
    },
    format: function (s, table, cell) {
      return $(cell).find('select').val() || s;
    },
    type: 'text'
  });

  $.tablesorter.addParser({
    id: 'input',
    is: function () {
      return false;
    },
    format: function (s, table, cell) {
      return $(cell).find(':checked').val() ? 1:0;
    },
    type: 'numerical'
  });

  // Update sorting on changes
  /* This is really slow with large nbr of developers, disable for now
  var alreadyUpdating = false;
  table.find('tbody').on('change', 'select, input', function (e) {
      if (!alreadyUpdating) {
          alreadyUpdating = true;
          $(this).trigger('update');
          setTimeout(function () {
              alreadyUpdating = false;
          }, 10);
      }
  });*/

  const headers = {
    '1': {sorter: 'select'},
    '2': {sorter: 'input'},
    '3': {sorter: 'input'},
    '4': {sorter: false}
  };

  if (!hasTeams) {
    headers['1'] = {sorter: 'input'};
    headers['3'] = {sorter: false};
  }

  tableSorter.sort(table, {
    headers: headers
  });
}
