// @flow
import $ from 'jquery';

import * as xhr from 'codescene-ui/src/browser/xhr';
import * as alerts from 'codescene-ui/src/alerts';
import * as fragmentationMap from 'codescene-ui/src/analyses/fragmentation-map';

import {renderPagedTable} from 'codescene-ui/src/paged-table';
import {showFragmentationModal} from './author-fragmentation';

function renderTable(container, data, projectId, analysisId, projectName, fractalSource, authorResolutionTitle) {
  const rows = data.map(entry => {
    function renderFractalsButton() {
      return $('<button>')
        .addClass('default-button small')
        .attr('title', `Show fractals of ${entry.entity}.`)
        .on('click', () => {
          showFragmentationModal(fractalSource, projectId, analysisId, entry.entity, {});
        })
        .html('<i class="fa fa-users">');
    }

    return [
      $('<span class="filename">').text(entry.entity),
      parseFloat(entry['fractal-value']).toLocaleString(),
      parseInt(entry['total-revs']).toLocaleString(),
      parseInt(entry['n-authors']).toLocaleString(),
      renderFractalsButton
    ];
  });

  renderPagedTable(container, rows, {
    title: `Most Fragmented Modules in ${projectName}`,
    headers: [
      { title: 'Module', class: 'column-module' },
      { title: 'Fragmentation <span>(0 - 1)</span>', class: 'column-fragmentation' },
      { title: 'Change Frequency', class: 'column-change-frequency' },
      { title: authorResolutionTitle, class: 'column-number-of-authors' },
      { title: '', class: 'column-actions' }
    ],
    sorting: {
      headers: {
        '4': {
          sorter: false
        }
      }
    }
  });
}

function indexByLeafIds(topSpots : Array<any>) {
  var indexed = {};

  function leafIdFromRow(d) {
    const name = d.entity.replace(/^.*[\\\/]/, '');
    const weight = parseFloat(d['fractal-value']).toFixed(2);
    const authors = d['n-authors'];
    return `${name}:${weight}:${authors}`;
  }

  topSpots.forEach(row => {
    indexed[leafIdFromRow(row)] = row;
  });

  return indexed;
}

function idFromCircle(d) {
  return `${d.name}:${d.weight.toFixed(2)}:${d.nauthors}`;
}

type ViewOptions = {
  moduleFragmentationUrl : string,
  mapFragmentationUrl : string,
  projectId : number,
  analysisId : number,
  projectName : string,
  sizeLabelCutOffPoint : number,
  fractalSource : string,
  type: string
};

export function init(options : ViewOptions) {
  const progressBar = alerts.progress($('.progress-container'), 'Loading...');
  const errorContainer = $('.error-container');
  const content = $('.content');
  const tableContainer = content.find('.module-fragmentation');
  const mapContainer = content.find('#fragmentation-map');
  if (mapContainer.length == 0) {
    progressBar.remove();
    content.show();
    return;
  }

  Promise.all([
    xhr.json(options.mapFragmentationUrl),
    xhr.csv(options.moduleFragmentationUrl)
  ]).then(([mapData, modules]) => {

    const modulesByLeafId = indexByLeafIds(modules);
    function onModuleClick (d) {
      const module = modulesByLeafId[idFromCircle(d)];
      if (module) {
        showFragmentationModal(options.fractalSource, options.projectId, options.analysisId, module.entity, {});
      }
    }

    fragmentationMap.drawMap(mapContainer, mapData, options.sizeLabelCutOffPoint, '#showLabelsCb', onModuleClick);
    renderTable(tableContainer, modules, options.projectId, options.analysisId, options.projectName, options.fractalSource, `Number of ${options.type}`);
    content.show();
  }).catch(err => {
    console.error(err.stack);
    alerts.warning(errorContainer, '<strong>Failed to load the fragmentation map.</strong> Try to run a new analysis.');
  }).then(() => {
    progressBar.remove();
  });
}
