// @flow
import $ from 'jquery';
import d3 from 'd3';

import * as xhr from 'codescene-ui/src/browser/xhr';
import * as alerts from 'codescene-ui/src/alerts';
import * as complexityTrends from 'codescene-ui/src/analyses/complexity-trends';

const parseDate = d3.time.format('%Y-%m-%d').parse;

export function parseRow(row : any) {
  return {
    comments: parseInt(row.comments),
    date: parseDate(row.date),
    dateString: row.date,
    max: parseInt(row.max),
    mean: parseFloat(row.mean),
    median: parseFloat(row.median),
    n: parseInt(row.n),
    sd: parseFloat(row.sd),
    total: parseFloat(row.total),
    ratio: parseFloat(row.ratio),
    revision: row.revision
  };
}

function onRevisionClicked(options, d, element, container) {
  var diffLinkHtml = '<a class="default-button full-width" target="_blank" href="' +
     window.baseUrl + options.projectId + '/analyses/' +
     options.analysisId +
     '/code/hotspots/complexity-trend/diff?name=' + options.fileName + '&revision=' + d.revision +
     '">Diff</a>';

  function closeContextMenu() {
    d3.select(container.get(0)).select('.context-menu').style('display', 'none');
  }

  d3.select(container.get(0)).select('.context-menu').remove();
  // create the div element that will hold the context menu
  d3.select(container.get(0)).selectAll('.context-menu').data([1])
    .enter()
    .append('div')
    .attr('class', 'context-menu');

  // this gets executed when a contextmenu event occurs
  var contextMenus = d3.select(container.get(0)).selectAll('.context-menu');

  contextMenus
    .on('click', function () {
      closeContextMenu();
    })
    .html('<div class="body">' +
    '<p><strong>' + 'Diff Revision ' + d.revision + '</strong></p>' +
    '</div>' +
    '<nav>' +
    diffLinkHtml +
    '</nav>');

  closeContextMenu();

  // show the context menu
  const [x, y] = d3.mouse(container.get(0));

  const menu = d3.select(container.get(0)).select('.context-menu')
    .style('top', (y + 10) + 'px')
    .style('display', 'block');

  menu.style('left', (x - 10) + 'px');
}

type ViewOptions = {
  complexityTrendUrl : string,
  projectId : number,
  analysisId : number,
  fileName : string
};

export function drawMainComplexityTrend(options : ViewOptions) {
  const body = $('.complexity-trend');
  xhr.csv(options.complexityTrendUrl, parseRow)
    .then(data => {
      if (data && data.length) {
        complexityTrends.drawComplexityTrend($('#trendline'), data, function(d, element, container) { return onRevisionClicked(options, d,element,container);});
      } else {
        alerts.warning(body, 'No complexity trend available - configure system level trends in your project.');
      }
    })
    .catch(err => {
      console.error(err);
      alerts.warning(body, 'No complexity trend available - configure system level trends in your project.');
    });
}

export function init(options : ViewOptions) {
  const body = $('.complexity-trend-view');
  xhr.csv(options.complexityTrendUrl, parseRow)
    .then(data => {
      complexityTrends.drawComplexityTrend($('#trendline'), data, function(d, element, $container) { return onRevisionClicked(options, d,element,$container);});
      complexityTrends.drawMaxTrend($('#maxline'), data);
      complexityTrends.drawMedianTrend($('#statsline'), data);
      complexityTrends.drawSdTrend($('#sdline'), data);

      const growthRatio = $('#growthratio');
      // The legacy format has no ratio.
      //
      if (data[0].ratio) {
        complexityTrends.drawRatioTrend(growthRatio, data);
      } else {
        growthRatio
          .addClass('text-muted')
          .html('This analysis has no data for complexity vs. lines of code. Re-run the analysis to see the results.');
      }
    })
    .catch(err => {
      console.error(err);
      alerts.error(body, 'Failed to get complexity trend data.');
    });
}
