// @flow
import $ from 'jquery';

import * as xhr from 'codescene-ui/src/browser/xhr';
import * as tabulator from 'codescene-ui/src/tabulator';
import * as alerts from 'codescene-ui/src/alerts';
import * as tableSorter from 'codescene-ui/src/table-sorter';

type ViewOptions = {
  churnByTask : string,
  pmIssues: String,
  ticketIdUriTemplate : string,
};

function renderTable(container, data, issues, ticketIdUriTemplate) {

  const hrefByTicketID = issues.reduce((m, x) => {
    m[x.id] = x.href;
    return m;
  }, {});

  data.forEach(function(d) {
    d.added = d.added.toLocaleString([], { useGrouping: true });
    d.deleted = d.deleted.toLocaleString([], { useGrouping: true });
    d.net = d.net.toLocaleString([], { useGrouping: true });
  });
  var fileTable = tabulator
    .tabulate(container.get(0), data, ['date','ticket','added','deleted','net','files','authors', 'leadtime'])
    .attr('id', 'churn-by-ticket-id-table')
    .classed('table-hover', true);

  fileTable.selectAll('thead th')
    .text(function(column, i) {
      if (i === 1) {
        return 'Task ID';
      } else if (i === 5) {
        return 'Changed Files';
      } else if (i === 6) {
        return 'Authors';
      } else if (i === 7) {
        return 'Lead Time (hours)';
      }

      return column.charAt(0).toUpperCase() + column.substr(1);
    });

  fileTable.selectAll('tbody td')
    .html(function(row) {
      if (row.column === 'ticket') {
        const href = hrefByTicketID[row.value];
        if (href) {
          return `<a href="${href}" title="View ticket ${row.value} in external system.">${row.value}</a>`;
        } else if (typeof ticketIdUriTemplate === 'string' && ticketIdUriTemplate !== '') {
          const uri = ticketIdUriTemplate.replace('{ticket-id}', row.value);
          return `<a href="${uri}" title="View ticket ${row.value} in external system.">${row.value}</a>`;
        } else {
          return row.value;
        }   
      } else {
        return row.value;
      }
    });

  tableSorter.sort($(fileTable.node()), {
    sortList: [[0, 1], [4, 1]]
  });
}

function parseLeadTimeWithBackwardsCompatibility(row) {
  const v = row.leadtime;

  if (v === undefined) {
    return 0;
  }

  return parseInt(v);
}

function parseRow(row) {
  return {
    date: row.date,
    ticket: row.ticket,
    added: parseInt(row.added),
    deleted: parseInt(row.deleted),
    net: parseInt(row.net),
    files: parseInt(row.files),
    authors: parseInt(row.authors),
    leadtime: parseLeadTimeWithBackwardsCompatibility(row)
  };
}

function parseIssueRow(row) {
  return {
    id: row.id,
    href: row.href
  };
}

export function init(options : ViewOptions) {
  const container = $('#tableview');

  xhr.csv(options.churnByTask, parseRow)
    .then(data => {
      if (data === undefined || data.length == 0) {
        const tableContainer = $('#tableview');
        alerts.warning(tableContainer, 'You need to configure a Ticket ID pattern in your project configuration to run this analysis.');
      } else {
        xhr.csv(options.pmIssues, parseIssueRow)
          .then(issues => {
            renderTable(container, data, issues, options.ticketIdUriTemplate);
          })
          .catch(() => {
            // No pm-issues available -> go on without them
            renderTable(container, data, [], options.ticketIdUriTemplate);
          });
      }
    })
    .catch(err => {
      console.error(err);
      alerts.error(container, 'Failed to get code churn by Ticket ID data.');
    });
}
