-- name: delete-all-analyses!
-- Deletes all existing analyses.
delete from analyses;

-- name: delete-analyses-for-project!
delete from analyses where project_id = ?;

-- name: delete-analysis-by-id!
delete from analyses where id = :analysis_id and project_id = :project_id;

-- name: delete-analysis-warning-by-analysis-id!
delete from warnings where analysis = :analysis_id and analysis IN(SELECT id from analyses where project_id = :project_id);

-- name: delete-warnings-for-project!
-- Removes all warning definitions.
delete from warnings where analysis IN(SELECT id from analyses where project_id = ?);

-- name: all-project-ids
-- Returns a sequence of all project IDs.
select id from projects;

-- name: all-analyses
-- Returns a sequence of all analyses, sorted by the latest analysis time.
select a.id, analysistime, c.rollingaveragedays, a.restricted,
a.ncomplexitytrends, a.firstanalysisdate, resultdir, duration, a.project_id, p.name, p.analysisdest from analyses as a
       join projects as p
            on p.id = a.project_id
       join project_analysis_configurations as c
            on p.config = c.id
       order by analysistime desc;

-- name: minimal-all-analyses
-- Like `all-analyses` but without any joins
select a.id, analysistime, a.restricted, a.ncomplexitytrends, a.firstanalysisdate, a.project_id  from analyses as a
       order by analysistime desc;

-- name: last-analysis-for-project
-- Returns the most recent analysis for the given project.
select a.id, analysistime, c.rollingaveragedays, a.restricted,
       a.ncomplexitytrends, a.firstanalysisdate, resultdir, duration, a.project_id, p.name, p.analysisdest
       from analyses as a
       join projects as p
            on p.id = a.project_id
       join project_analysis_configurations as c
            on p.config = c.id
       where p.id = ?
       order by analysistime desc
       limit 1;

-- name: last-analysis-for-projects
-- Returns the most recent analysis for every project
-- This is an optimized version of `last-analysis-for-project`
-- for cases where we are interested in all the projects such as when handing `run-status` API call
select a.id, analysistime, c.rollingaveragedays, a.restricted,
       a.ncomplexitytrends, a.firstanalysisdate, resultdir, duration, a.project_id, p.name, p.analysisdest
  from analyses as a
       join projects as p on p.id = a.project_id
       join project_analysis_configurations as c on p.config = c.id
 where a.id = (SELECT aa.id from analyses aa where aa.PROJECT_ID = p.id order by analysistime desc limit 1);

-- name: older-analyses-for-project
-- Returns the analyses that are older than the given date/time, except for the latest completed analysis
select a.id, analysistime, resultdir, a.project_id, p.analysisdest from analyses as a
       join projects as p
            on p.id = a.project_id
       where a.project_id = ? AND
             analysistime < :oldesttime AND
             a.id != (SELECT MAX(id) from analyses where PROJECT_ID = p.id );

-- name: analysis-by-id
-- Returns a specific analysis.
select a.id, analysistime, c.rollingaveragedays, a.restricted,
a.firstanalysisdate, a.ncomplexitytrends,
       resultdir, duration,
       p.id as project_id, p.name, p.analysisdest
       from analyses as a
       join projects as p
            on p.id = a.project_id
       join project_analysis_configurations as c
            on p.config = c.id
       where p.id = :project_id and a.id = :analysis_id;

-- name: analysis-by-project-id
-- Returns all analyses for the given project.
select a.id, analysistime, a.firstanalysisdate, resultdir, duration, c.rollingaveragedays,
a.project_id, a.ncomplexitytrends, p.name, p.analysisdest, a.restricted from analyses as a
   join projects as p
            on p.id = a.project_id
       join project_analysis_configurations as c
            on p.config = c.id
       where p.id = ?
       order by analysistime desc;

-- name: minimal-analysis-by-project-id
-- Returns the list of analyses id and name for given project id
select a.id, p.name, a.analysistime from analyses as a, projects as p
    where p.id = a.project_id and p.id = :project_id
    order by a.id desc;

-- name: insert-analysis<!
-- Inserts the given analysis while returning its generated ID.
insert into analyses
       (analysistime, firstanalysisdate, resultdir, duration, project_id, ncomplexitytrends, restricted)
       values(:analysistime, :firstanalysisdate, :resultdir, :duration, :project_id, :ncomplexitytrends, :restricted)

-- name: update-result-path!
--- Updates the thresholds for rising spots warnings.
update analyses
  set resultdir = :resultdir
     where id = ?;

-- name: repository-revisions-by-analysis-id
-- Retrieves the revisions for a given analysis of each repository.
select analysis_vcs_revision, repo
       from analysis_repository_revisions
       where analysis_id = :analysis_id;

-- name: insert-repository-revisions!
-- Records the revisions of each repository in the analysis.
insert into analysis_repository_revisions
    (analysis_vcs_revision, repo, analysis_id)
    values(:analysis_vcs_revision, :repo, :analysis_id)
